class IndexedDBHelper {
    constructor(dbName, dbVersion) {
        this.dbName = dbName;
        this.dbVersion = dbVersion;
        this.db = null;
    }

    /**
     * 打开数据库
     */
    openDatabase(upgradeCallback) {
        if (this.db) return this.db; // 已初始化，不重复打开
        return new Promise((resolve, reject) => {
            const request = indexedDB.open(this.dbName, this.dbVersion);

            request.onsuccess = (event) => {
                this.db = event.target.result;
                console.log("Database opened successfully");
                resolve(this.db);
            };

            request.onerror = (event) => {
                console.error("Database error:", event.target.error);
                reject(event.target.error);
            };

            request.onupgradeneeded = (event) => {
                console.log("Database upgrade needed");
                const db = event.target.result;
                if (typeof upgradeCallback === "function") {
                    upgradeCallback(db, event);
                }
            };
        });
    }

    deleteDatabase() {
      return new Promise((resolve, reject) => {
        const request = indexedDB.deleteDatabase('gyws'); // 'gyws' 是数据库的名字

        request.onsuccess = function(event) {
          console.log("数据库删除成功");
          resolve(event); // 成功时，解析 Promise
        };

        request.onerror = function(event) {
          console.error("数据库删除失败", event.target.error);
          reject(event.target.error); // 失败时，拒绝 Promise
        };

        request.onblocked = function(event) {
          console.warn("数据库删除被阻塞");
          reject(event.target.error); // 阻塞时拒绝 Promise
        };
      });
    }


    /**
     * 添加数据
     */
    addData(storeName, data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);
            const request = store.add(data);

            request.onsuccess = () => {
                console.log("Data added successfully");
                console.log(data);
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error adding data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 获取数据
     */
    getData(storeName, key) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readonly");
            const store = transaction.objectStore(storeName);
            const request = store.get(key);

            request.onsuccess = () => {
                console.log("Data retrieved successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error retrieving data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 获取数据
     */
    getFilterData(storeName, filter_data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readonly");
            const store = transaction.objectStore(storeName);
            const request = store.getAll();

            request.onsuccess = () => {
                console.log("All data retrieved successfully");
                const allData = request.result;
                // 遍历数据并筛选
                const filteredData = allData.filter(item => {
                    // 遍历 filter 的每个字段做比对
                    let price_min = filter_data['price_min'];
                    let price_max = filter_data['price_max'];
                    let sale_min = filter_data['sale_min'];
                    let sale_max = filter_data['sale_max'];
                    //
                    if(price_max > 0){
                        if(item['_addition_data']['_price'] > price_max){
                            return false;
                        }
                    }
                    if(price_min >0 ){
                        if(item['_addition_data']['_price'] < price_min){
                            return false;
                        }
                    }
                    if(sale_max > 0){
                        if(item['_addition_data']['_sold'] > sale_max){
                            return false;
                        }
                    }
                    if(sale_min > 0){
                        if(item['_addition_data']['_sold'] < sale_min){
                            return false;
                        }
                    }
                    return true; // 所有字段都匹配，保留该项
                });
                //
                resolve(filteredData);
            };

            request.onerror = (event) => {
                console.error("Error retrieving all data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 获取所有数据
     */
    getAllData(storeName) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readonly");
            const store = transaction.objectStore(storeName);
            const request = store.getAll();

            request.onsuccess = () => {
                console.log("All data retrieved successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error retrieving all data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 更新数据
     */
    updateData(storeName, data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);
            const request = store.put(data);

            request.onsuccess = () => {
                console.log("Data updated successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error updating data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    updateDataById(storeName, id, updatedFields){
        return new Promise((resolve, reject) =>{
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);

            //获取当前id存储对象
            const getRequest = store.get(id);

            getRequest.onsuccess = (event) => {
                const record = event.target.result;
                if(record){
                    //合并更细字段
                    const updatedRecord = { ...record, ...updatedFields };
                    // 更新记录
                    const updateRequest = store.put(updatedRecord);
                    //
                    updateRequest.onsuccess = () => {
                        console.log("Record updated successfully:", updatedRecord);
                        resolve(updatedRecord);
                    };

                    updateRequest.onerror = (error) => {
                        console.error("Error updating record:", error);
                        reject(error);
                    };
                }else{
                    //
                    console.error(`Record with id ${id} not found.`);
                    reject(new Error(`Record with id ${id} not found.`));
                    //
                }
            }

            getRequest.onerror = (event) => {
                console.error("Error fetching record:", error);
                reject(error);
            }
        });
    }

    /**
     * 删除数据
     */
    deleteData(storeName, key) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);
            const request = store.delete(key);

            request.onsuccess = () => {
                console.log("Data deleted successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error deleting data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 清空对象存储
     */
    clearStore(storeName) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);
            const request = store.clear();

            request.onsuccess = () => {
                console.log("Object store cleared successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error clearing object store:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 清空已采集 针对products
     */
    clearProductsHas(){
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction(['products'], "readwrite");
            const store = transaction.objectStore('products');
            const index = store.index('_is_request_index');  // 获取索引

            const cursorRequest = index.openCursor(IDBKeyRange.only(2));

            cursorRequest.onsuccess = (event) => {
                const cursor = event.target.result;

                if (cursor) {
                    // 如果找到符合条件的记录，删除它
                    console.log('cursor.primaryKey:'+cursor.primaryKey); // 获取主键
                    store.delete(cursor.primaryKey);
                    console.log(`Deleted item with primaryKey: ${cursor.primaryKey}`);
                    cursor.continue();  // 继续遍历下一条记录
                } else {
                    console.log("All matching records deleted.");
                    resolve();
                }
            };

            cursorRequest.onerror = (event) => {
                console.error("Error during cursor iteration:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 查询符合采集条件的产品
     */
    findNeedSpiderProduct(storeName){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName,'videos'], 'readonly');
            let store = transaction.objectStore(storeName);
            let otherStore = transaction.objectStore('videos');
            let index = store.index('_is_request_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.openCursor(1); //查询 _is_request = 1 且倒序
            request.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findNeedSpiderProduct data:"+ cursor.value);
                    let item =  cursor.value;
                    if(item['_s_time'] < (Date.now()-30000)){ //60000
                        resolve(cursor.value);
                    }else{
                        resolve(null);
                    }
                }
            };
            request.onerror = (event) => {
                console.error("findNeedSpiderProduct Error:", event.target.error);
                reject('error');
            }
            //
            let request2 = index.openCursor(0); //查询 _is_request = 0 且倒序
            request2.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findNeedSpiderProduct data:"+ cursor.value);
                    let item =  cursor.value;
                    //
                    //从第二个表中查找对应数据，例如根据 product_id 查询
                    let otherIndex = otherStore.index('_channel_key_index'); // 需确保你已创建这个索引
                    let otherRequest = otherIndex.getAll(item._product_key);
                    otherRequest.onsuccess = (e2) => {
                        let relatedRecords = e2.target.result;
                        console.log("relatedRecords:", relatedRecords);
                        //
                        if(relatedRecords.length>0){
                            //
                            for(let _video of relatedRecords){
                                if(_video['_video_is_request'] == 2){
                                    resolve(item);
                                }
                            }
                            resolve(null);
                            //
                        }else{
                            resolve(item);
                        }
                    };
                    otherRequest.onerror = (e2) => {
                        console.error("Error querying otherStore:", e2.target.error);
                        reject(e2.target.error);
                    };
                    //
                }else{
                    resolve(null);
                }
            };
            request2.onerror = (event) => {
                console.error("findNeedSpiderProduct Error:", event.target.error);
                reject('error');
            }
            //
        });
    }

    //查询总记录数
    getRecordCount(storeName){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName], 'readonly');
            let store = transaction.objectStore(storeName);
            //
            const request = store.count();

            request.onsuccess = (event) => {
                resolve(event.target.result); // 返回总记录数
            }

            request.onerror = (event) => {
                reject(event.target.error); // 捕获错误
            }
        });
    }

    //查询总记录数
    getRecordCountOfYoutube(storeName){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName], 'readonly');
            let store = transaction.objectStore(storeName);
            // 获取索引
            const index = store.index('_platform_index');
            // 统计索引值为 'youtube' 的记录数量
            const request = index.count('youtube');

            request.onsuccess = (event) => {
                resolve(event.target.result); // 返回总记录数
            }

            request.onerror = (event) => {
                reject(event.target.error); // 捕获错误
            }
        });
    }

    //查询已完成的记录数
    //针对products
    getProductsHasRecordCount(){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['products'], 'readonly');
            let store = transaction.objectStore('products');
            let count = 0;
            const request = store.openCursor();
            request.onsuccess = (event) => {
                const cursor = event.target.result;
                if (cursor) {
                    const record_data = cursor.value;
                    // 满足条件则计数
                    if (record_data['_is_request'] == 2) {
                        count++;
                    }
                    cursor.continue(); // 继续遍历
                } else {
                    // 遍历结束
                    resolve(count);
                }
            };
            request.onerror = (event) => {
                reject(event.target.error);
            };
        });
    }

    //查询已完成的记录数
    //针对videos
    getVidoesHasRecordCount(){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['videos'], 'readonly');
            let store = transaction.objectStore('videos');
            let count = 0;
            const request = store.openCursor();
            request.onsuccess = (event) => {
                const cursor = event.target.result;
                if (cursor) {
                    const record_data = cursor.value;
                    // 满足条件则计数
                    if (record_data['_video_is_request'] == 2) {
                        count++;
                    }
                    cursor.continue(); // 继续遍历
                } else {
                    // 遍历结束
                    resolve(count);
                }
            };
            request.onerror = (event) => {
                reject(event.target.error);
            };
        });
    }

    //查询已完成的记录数
    //针对videos
    getVidoesHasRecordCountOfYoutube(){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['videos'], 'readonly');
            let store = transaction.objectStore('videos');
            //
            const index = store.index('_platform_index'); // 按平台索引查询
            const request = index.openCursor('youtube'); // 只遍历 platform = youtube 的记录

            let count = 0;
            request.onsuccess = (event) => {
                const cursor = event.target.result;
                if (cursor) {
                    const record_data = cursor.value;
                    // 满足条件则计数
                    if (record_data['_video_is_request'] == 2) {
                        count++;
                    }
                    cursor.continue(); // 继续遍历
                } else {
                    // 遍历结束
                    resolve(count);
                }
            };
            request.onerror = (event) => {
                reject(event.target.error);
            };
        });
    }

    //查询已完成的记录数
    //针对videos
    getVidoesHasWhatsapp(){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['videos'], 'readonly');
            let store = transaction.objectStore('videos');
            let whatsapp_group_num = 0;
            let whatsapp_number_num = 0;
            const request = store.openCursor();
            request.onsuccess = (event) => {
                const cursor = event.target.result;
                if (cursor) {
                    const record_data = cursor.value;
                    // 满足条件则计数
                    if (record_data['_video_is_request'] == 2) {
                        //
                        if(record_data['_hongren_whatsapp_group']){
                            whatsapp_group_num++;
                        }
                        if(record_data['_hongren_whatsapp']){
                            whatsapp_number_num++;
                        }
                        //
                    }
                    cursor.continue(); // 继续遍历
                } else {
                    // 遍历结束
                    resolve({
                        'whatsapp_group_num':whatsapp_group_num,
                        'whatsapp_number_num':whatsapp_number_num
                    });
                }
            };
            request.onerror = (event) => {
                reject(event.target.error);
            };
        });
    }

    /**
     * 查询符合采集条件的产品
     */
    getProductBy_product_key(_product_key){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['products'], 'readonly');
            let store = transaction.objectStore('products');
            let index = store.index('_product_key_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.get(_product_key);
            request.onsuccess = (event) => {
                let result  = event.target.result;
                if(result){
                    console.log("getProductBy_product_key data:"+ result);
                    resolve(result);
                }else {
                    resolve(null);  // 没找到
                }
            };
            request.onerror = (event) => {
                console.error("getProductBy_product_key Error:", event.target.error);
                reject('error');
            }
            //
        });
    }

    /**
     * 查询符合采集条件的domain
     */
    findNeedSpiderDomain(){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['products'], 'readonly');
            let store = transaction.objectStore('products');
            let index = store.index('_companywebsite_emails_is_request_and_is_request_index');
            let keyRange = IDBKeyRange.only([2, 0]);
            let request = index.openCursor(keyRange);
            request.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    resolve(cursor.value);
                }else{
                    resolve(null);
                }
            };
            request.onerror = (event) => {
                console.error("findNeedSpiderProduct Error:", event.target.error);
                reject('error');
            }
        });
    }

    /**
     * 查询符合call gycharm
     */
    findNeedCallgycharmProduct(storeName){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName], 'readonly');
            let store = transaction.objectStore(storeName);
            let index = store.index('_isrequest_callgycharm_index');
            let keyRange = IDBKeyRange.only([2, 2, 0]);
            let request = index.openCursor(keyRange);
            request.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    resolve(cursor.value);
                }else{
                    resolve(null);
                }
            };
            request.onerror = (event) => {
                console.error("findNeedSpiderProduct Error:", event.target.error);
                reject('error');
            }
        });
    }

    /**
     * 查询符合采集条件的产品
     */
    findNeedSpiderTask(storeName){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName], 'readonly');
            let store = transaction.objectStore(storeName);
            let index = store.index('_is_request_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.openCursor(1); //查询 _is_request = 1 且倒序
            request.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findNeedSpiderTask data:"+ cursor.value);
                    let item =  cursor.value;
                    if(item['_s_time'] < (Date.now()-60000*10)){ //60000
                        resolve(cursor.value);
                    }else{
                        resolve(null);
                    }
                }
            };
            request.onerror = (event) => {
                console.error("findNeedSpiderTask Error:", event.target.error);
                reject('error');
            }
            //
            let request2 = index.openCursor(0); //查询 _is_request = 0 且倒序
            request2.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findNeedSpiderTask data:"+ cursor.value);
                    let item =  cursor.value;
                    resolve(item);
                }else{
                    resolve(null);
                }
            };
            request2.onerror = (event) => {
                console.error("findNeedSpiderTask Error:", event.target.error);
                reject('error');
            }
            //
        });
    }

     /**
     * 查询符合采集条件的产品
     */
    getTaskBy_product_key(_product_key){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['tasks'], 'readonly');
            let store = transaction.objectStore('tasks');
            let index = store.index('_product_key_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.get(_product_key);
            request.onsuccess = (event) => {
                let result  = event.target.result;
                if(result){
                    console.log("getTaskBy_product_key data:"+ result);
                    resolve(result);
                }else {
                    resolve(null);  // 没找到
                }
            };
            request.onerror = (event) => {
                console.error("getTaskBy_product_key Error:", event.target.error);
                reject('error');
            }
            //
        });
    }

      /**
     * 查询符合采集条件的产品
     */
    findNeedSpiderVideo(storeName){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName], 'readonly');
            let store = transaction.objectStore(storeName);
            let index = store.index('_video_is_request_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.openCursor(1); //查询 _is_request = 1 且倒序
            request.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findNeedSpiderVideo data:"+ cursor.value);
                    let item =  cursor.value;
                    if(item['_s_time'] < (Date.now()-30000)){ //60000
                        resolve(cursor.value);
                    }else{
                        resolve(null);
                    }
                }
            };
            request.onerror = (event) => {
                console.error("findNeedSpiderVideo Error:", event.target.error);
                reject('error');
            }
            //
            let request2 = index.openCursor(0); //查询 _is_request = 0 且倒序
            request2.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findNeedSpiderVideo data:"+ cursor.value);
                    let item =  cursor.value;
                    resolve(item);
                }else{
                    resolve(null);
                }
            };
            request2.onerror = (event) => {
                console.error("findNeedSpiderVideo Error:", event.target.error);
                reject('error');
            }
            //
        });
    }

    /**
     * 获取所有数据
     */
    getAllDataVideosAndKehu() {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction(['videos', 'products'], "readonly");
            const videosStore = transaction.objectStore('videos');
            const kehuStore = transaction.objectStore('products');

            // 先缓存 kehu 表数据
            const kehuMap = {};
            const kehuRequest = kehuStore.getAll();
            const results = [];

            kehuRequest.onsuccess = () => {
                (kehuRequest.result || []).forEach(k => kehuMap[k._product_key] = k);

                const cursorRequest = videosStore.openCursor();
                cursorRequest.onsuccess = (event) => {
                    const cursor = event.target.result;
                    if (cursor) {
                        const v = cursor.value;
                        const kehu = kehuMap[v._channel_key];
                        results.push({
                            ...v,
                            kehu_data: kehu ? kehu._product_data : null,
                            kehu_companywebsite_emails_is_request: kehu ? kehu._companywebsite_emails_is_request : 0,
                            kehu_is_request: kehu ? kehu._is_request :0
                        });
                        cursor.continue();
                    } else {
                        console.log("关联查询结果：", results);
                        resolve(results);
                    }
                };

                cursorRequest.onerror = (e) => reject(e.target.error);
            };

            kehuRequest.onerror = (e) => reject(e.target.error);
        });
    }

    /**
     *
     */
    findVideoBy_channel_key(_channel_key){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['videos'], 'readonly');
            let store = transaction.objectStore('videos');
            let index = store.index('_channel_key_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.openCursor(_channel_key);
            request.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                   resolve(cursor.value);
                }else{
                    resolve(null);
                }
            };
            request.onerror = (event) => {
                console.error("findNeedSpiderProduct Error:", event.target.error);
                reject('error');
            }
            //
        });
    }


}