class IndexedDBHelper {
    constructor(dbName, dbVersion) {
        this.dbName = dbName;
        this.dbVersion = dbVersion;
        this.db = null;
    }

    /**
     * 打开数据库
     */
    openDatabase(upgradeCallback) {
        return new Promise((resolve, reject) => {
            const request = indexedDB.open(this.dbName, this.dbVersion);

            request.onsuccess = (event) => {
                this.db = event.target.result;
                console.log("Database opened successfully");
                resolve(this.db);
            };

            request.onerror = (event) => {
                console.error("Database error:", event.target.error);
                reject(event.target.error);
            };

            request.onupgradeneeded = (event) => {
                console.log("Database upgrade needed");
                const db = event.target.result;
                if (typeof upgradeCallback === "function") {
                    upgradeCallback(db, event);
                }
            };
        });
    }

    deleteDatabase() {
      return new Promise((resolve, reject) => {
        const request = indexedDB.deleteDatabase('gy1688'); // 'gy1688' 是数据库的名字

        request.onsuccess = function(event) {
          console.log("数据库删除成功");
          resolve(event); // 成功时，解析 Promise
        };

        request.onerror = function(event) {
          console.error("数据库删除失败", event.target.error);
          reject(event.target.error); // 失败时，拒绝 Promise
        };

        request.onblocked = function(event) {
          console.warn("数据库删除被阻塞");
          reject(event.target.error); // 阻塞时拒绝 Promise
        };
      });
    }


    /**
     * 添加数据
     */
    addData(storeName, data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);
            const request = store.add(data);

            request.onsuccess = () => {
                console.log("Data added successfully");
                console.log(data);
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error adding data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 获取数据
     */
    getData(storeName, key) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readonly");
            const store = transaction.objectStore(storeName);
            const request = store.get(key);

            request.onsuccess = () => {
                console.log("Data retrieved successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error retrieving data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 获取数据
     */
    getFilterData(storeName, filter_data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readonly");
            const store = transaction.objectStore(storeName);
            const request = store.getAll();

            request.onsuccess = () => {
                console.log("All data retrieved successfully");
                const allData = request.result;
                // 遍历数据并筛选
                const filteredData = allData.filter(item => {
                    // 遍历 filter 的每个字段做比对
                    let price_min = filter_data['price_min'];
                    let price_max = filter_data['price_max'];
                    let sale_min = filter_data['sale_min'];
                    let sale_max = filter_data['sale_max'];
                    //
                    if(price_max > 0){
                        if(item['_addition_data']['_price'] > price_max){
                            return false;
                        }
                    }
                    if(price_min >0 ){
                        if(item['_addition_data']['_price'] < price_min){
                            return false;
                        }
                    }
                    if(sale_max > 0){
                        if(item['_addition_data']['_sold'] > sale_max){
                            return false;
                        }
                    }
                    if(sale_min > 0){
                        if(item['_addition_data']['_sold'] < sale_min){
                            return false;
                        }
                    }
                    return true; // 所有字段都匹配，保留该项
                });
                //
                resolve(filteredData);
            };

            request.onerror = (event) => {
                console.error("Error retrieving all data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 获取所有数据
     */
    getAllData(storeName) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readonly");
            const store = transaction.objectStore(storeName);
            const request = store.getAll();

            request.onsuccess = () => {
                console.log("All data retrieved successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error retrieving all data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 更新数据
     */
    updateData(storeName, data) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);
            const request = store.put(data);

            request.onsuccess = () => {
                console.log("Data updated successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error updating data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    updateDataById(storeName, id, updatedFields){
        return new Promise((resolve, reject) =>{
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);

            //获取当前id存储对象
            const getRequest = store.get(id);

            getRequest.onsuccess = (event) => {
                const record = event.target.result;
                if(record){
                    //合并更细字段
                    const updatedRecord = { ...record, ...updatedFields };
                    // 更新记录
                    const updateRequest = store.put(updatedRecord);
                    //
                    updateRequest.onsuccess = () => {
                        console.log("Record updated successfully:", updatedRecord);
                        resolve(updatedRecord);
                    };

                    updateRequest.onerror = (error) => {
                        console.error("Error updating record:", error);
                        reject(error);
                    };
                }else{
                    //
                    console.error(`Record with id ${id} not found.`);
                    reject(new Error(`Record with id ${id} not found.`));
                    //
                }
            }

            getRequest.onerror = (event) => {
                console.error("Error fetching record:", error);
                reject(error);
            }
        });
    }

    /**
     * 删除数据
     */
    deleteData(storeName, key) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);
            const request = store.delete(key);

            request.onsuccess = () => {
                console.log("Data deleted successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error deleting data:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 清空对象存储
     */
    clearStore(storeName) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([storeName], "readwrite");
            const store = transaction.objectStore(storeName);
            const request = store.clear();

            request.onsuccess = () => {
                console.log("Object store cleared successfully");
                resolve(request.result);
            };

            request.onerror = (event) => {
                console.error("Error clearing object store:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 清空已采集 针对products
     */
    clearProductsHas(){
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction(['products'], "readwrite");
            const store = transaction.objectStore('products');
            const index = store.index('_is_request_index');  // 获取索引

            const cursorRequest = index.openCursor(IDBKeyRange.only(2));

            cursorRequest.onsuccess = (event) => {
                const cursor = event.target.result;

                if (cursor) {
                    // 如果找到符合条件的记录，删除它
                    console.log('cursor.primaryKey:'+cursor.primaryKey); // 获取主键
                    store.delete(cursor.primaryKey);
                    console.log(`Deleted item with primaryKey: ${cursor.primaryKey}`);
                    cursor.continue();  // 继续遍历下一条记录
                } else {
                    console.log("All matching records deleted.");
                    resolve();
                }
            };

            cursorRequest.onerror = (event) => {
                console.error("Error during cursor iteration:", event.target.error);
                reject(event.target.error);
            };
        });
    }

    /**
     * 查询符合采集条件的产品
     */
    findNeedSpiderProduct(storeName){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName], 'readonly');
            let store = transaction.objectStore(storeName);
            let index = store.index('_is_request_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.openCursor(1); //查询 _is_request = 1 且倒序
            request.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findNeedSpiderProduct data:"+ cursor.value);
                    let item =  cursor.value;
                    if(item['_s_time'] < (Date.now()-30000)){ //60000
                        resolve(cursor.value);
                    }else{
                        resolve(null);
                    }
                }
            };
            request.onerror = (event) => {
                console.error("findNeedSpiderProduct Error:", event.target.error);
                reject('error');
            }
            //
            let request2 = index.openCursor(0); //查询 _is_request = 0 且倒序
            request2.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findNeedSpiderProduct data:"+ cursor.value);
                    let item =  cursor.value;
                    resolve(item);
                }else{
                    resolve(null);
                }
            };
            request2.onerror = (event) => {
                console.error("findNeedSpiderProduct Error:", event.target.error);
                reject('error');
            }
            //
        });
    }

    /**
     * 查询符合采集条件的产品
     */
    checkIsrepeat(storeName, _seller_name=''){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName], 'readonly');
            let store = transaction.objectStore(storeName);
            let index = store.index('_is_request_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.openCursor(2); //查询 _is_request = 2 且倒序
            request.onsuccess = (event) => {
                let cursor = event.target.result;
                if(cursor){
                    console.log("findBySellername data:"+ cursor.value);
                    let item =  cursor.value;
                    if(item['_seller_name'] == _seller_name){ //60000
                        resolve(1);
                    }else{
                        cursor.continue();  // 继续遍历下一条记录
                    }
                }
                resolve(0);
            };
            request.onerror = (event) => {
                console.error("findNeedSpiderProduct Error:", event.target.error);
                reject('error');
            }
            //
        });
    }

    //查询总记录数
    getRecordCount(storeName){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction([storeName], 'readonly');
            let store = transaction.objectStore(storeName);
            //
            const request = store.count();

            request.onsuccess = (event) => {
                resolve(event.target.result); // 返回总记录数
            }

            request.onerror = (event) => {
                reject(event.target.error); // 捕获错误
            }
        });
    }

    //查询已完成的记录数
    //针对products
    getProductsHasRecordCount(){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['products'], 'readonly');
            let store = transaction.objectStore('products');
            let count = 0;
            const request = store.openCursor();
            request.onsuccess = (event) => {
                const cursor = event.target.result;
                if (cursor) {
                    const record_data = cursor.value;
                    // 满足条件则计数
                    if (record_data['_is_request'] == 2) {
                        count++;
                    }
                    cursor.continue(); // 继续遍历
                } else {
                    // 遍历结束
                    resolve(count);
                }
            };
            request.onerror = (event) => {
                reject(event.target.error);
            };
        });
    }

    /**
     * 查询符合采集条件的产品
     */
    getProductBy_product_key(_product_key){
        return new Promise((resolve, reject) => {
            let transaction = this.db.transaction(['products'], 'readonly');
            let store = transaction.objectStore('products');
            let index = store.index('_product_key_index');
             // 使用索引游标按倒序（prev）遍历 ,目前用升序
            let request = index.get(_product_key);
            request.onsuccess = (event) => {
                let result  = event.target.result;
                if(result){
                    console.log("getProductBy_product_key data:"+ result);
                    resolve(result);
                }else {
                    resolve(null);  // 没找到
                }
            };
            request.onerror = (event) => {
                console.error("getProductBy_product_key Error:", event.target.error);
                reject('error');
            }
            //
        });
    }

}